# @types/sighthoundinc

Comprehensive TypeScript type definitions for Sighthound Redactor and related APIs.

## Overview

This is a zero-dependency, types-only module providing complete TypeScript definitions for:

- **Redactor Embedded Editor** - Browser-based video redaction interface
- **Analytics** - Object detection (faces, license plates, vehicles, documents)
- **Metadata** - Custom and system metadata management
- **API Types** - Core API request/response structures
- **Logging** - Configurable logging levels

## Installation

This package comes distributed as a ZIP file:

1. Extract to your project directory
2. Install with path reference:

```bash
npm install ./path/to/sh-types
```

3. Or add to `tsconfig.json` without installation:

```json
{
  "compilerOptions": {
    "typeRoots": ["./node_modules/@types", "./path/to/sh-types"]
  }
}
```

## Usage

### Basic Import

```typescript
import "@types/sighthoundinc";

// Types are now available in global namespace
const editor: Sighthound.Redactor.IEditor = ...;
```

### Embedded Redactor Editor

```typescript
const container = document.getElementById("redactor-container");

const editor = new Sighthound.Redactor.SighthoundRedactorEditor(
  container,
  "https://your-redactor-server.com",
  sessionId,
  {
    locale: "en-US",
    logLevel: 2, // Info level
    shadowed: true,
    iframed: false,
    customize: {
      export: { download: "show", save: "close" },
      session: { auditLogs: "show", folders: "hide" }
    }
  }
);

await editor.renderEditorAsync();
```

### Analytics Types

```typescript
const detection: Sighthound.Analytics.IRawObject = {
  id: 1,
  uuid: "550e8400-e29b-41d4-a716-446655440000",
  type: "vehicle",
  firstFrame: 0,
  lastFrame: 100,
  bestFrame: 50,
  attributes: {
    type: "car",
    make: { name: "Toyota", confidence: 0.95 },
    color: { name: "blue", confidence: 0.88 }
  }
};
```

### Metadata Management

```typescript
const metadata: Sighthound.Metadata.Collection<unknown> = {
  sio_file_name: "video.mp4",
  sio_file_size: 1024000,
  sio_file_sha256: "abc123...",
  etc_media_title: "Evidence Video",
  etc_media_path: ["Case-123", "Evidence"],
  custom_case_number: "2024-001",
  custom_case_number__en_US: "Case Number"
};
```

## Type Definitions

### Core Modules

| Module | Description |
|--------|-------------|
| `redactor.d.ts` | Embedded editor interface, options, commands |
| `analytics.d.ts` | Object detection types and confidence values |
| `metadata.d.ts` | System, custom, and event metadata |
| `logging.d.ts` | Log levels (0=trace → 5=off) |
| `api/` | API versioning and request/response types |

### Key Interfaces

#### `Sighthound.Redactor.IEditor`

Main editor interface with methods:
- `active()` - Check if video is loaded
- `cleanup()` - Close current video
- `destroy()` - Cleanup instance
- `renderEditor()` / `renderEditorAsync()` - Mount editor
- `sendCommand()` - Execute commands (export, close, destroy)
- `setLocale()` - Change language

#### `Sighthound.Redactor.IEditorOptions`

Configuration options:
- `locale` - Language (default: "en-US")
- `logLevel` - Console output verbosity (0-5)
- `shadowed` - Use Shadow DOM isolation
- `iframed` - Host in iframe
- `customize` - Show/hide UI elements
- `cssHref` - Custom styling
- `translationOverwritesUrl` - Custom translations

#### `Sighthound.Analytics.AutoDetectType`

Detectable object types:
- `face` / `head` - Human faces
- `person` - Full person detection
- `vehicle` - Cars, trucks, buses, motorcycles
- `licenseplate` - License plate OCR
- `document` - Documents and papers
- `screen` - Monitors, laptops, mobile devices
- `ID` - Identification cards

#### `Sighthound.Metadata.Collection`

Metadata structure:
- `sio_*` - System metadata (file info, hashes, duration)
- `etc_*` - Event/configuration metadata
- `custom_*` - User-defined metadata with i18n support

## API Versions

Supported API versions:
- **v1** - Original API
- **v11** - Recent updates (backwards compatible)

Products:
- `core` - Core processing
- `cloud` - Cloud services
- `redactor` - Redactor-specific

## Customization

### UI Element Visibility

```typescript
const customize: Sighthound.Redactor.ICustomize = {
  export: {
    download: "show",      // Show download button
    save: "eventOnly"      // Fire event but no UI
  },
  session: {
    auditLogs: "hide",     // Hide audit log viewer
    folders: "show",       // Show folder organization
    remove: "eventOnly",   // Custom removal handling
    deleteRedactionData: "hide"
  },
  videolist: {
    customMetadataKeys: ["custom_case_number", "custom_officer"]
  }
};
```

Options: `"show"` | `"hide"` | `"eventOnly"` | `"close"`

### Custom Styling

```typescript
cssHref: {
  main: [
    "https://your-cdn.com/redactor-overrides.css",
    "/local/styles.css"
  ],
  fonts: "https://fonts.googleapis.com/css2?family=Inter"
}
```

## Advanced Features

### Shadow DOM Isolation

Recommended for embedding in complex applications:

```typescript
const editor = new SighthoundRedactorEditor(container, serverUrl, sessionId, {
  shadowed: true,  // Isolates styles and events
  iframed: false
});
```

### Server Logging

Send client logs to backend:

```typescript
logToServer: true,
logPrefix: "[ClientApp]",
logLevel: 1  // Debug
```

### Command Execution

```typescript
// Trigger export dialog
editor.sendCommand("export");

// Close current video
editor.sendCommand("close");

// Cleanup and destroy
editor.sendCommand("destroy");
```

## Browser Compatibility

Requires modern browsers with:
- ES6+ support
- Shadow DOM v1 (if using `shadowed: true`)
- Web Components (Custom Elements)

## License

Copyright © 2026 Sighthound, Inc. All rights reserved.

Proprietary and confidential. No part of this software may be used, reproduced, stored or distributed without explicit written authorization.

## Support

For issues or questions, contact Sighthound support or refer to the [developer documentation](https://dev.sighthound.com/redactor/overview/).
