/* ----------------------------------------------------------------------
 * Copyright (C) 2018 Sighthound, Inc. All rights reserved.
 * The information and source code contained herein is the
 * exclusive property of Sighthound, Inc. No part of this software
 * may be used, reproduced, stored or distributed in any form,
 * without explicit written authorization from Sighthound, Inc.
 * ---------------------------------------------------------------------- */

declare namespace Sighthound.Analytics {

/**
 * [current labels](https://github.com/sighthoundinc/DeepModel_MultiObjectDetection/blob/develop/17classes_document/gen7es/labels.txt)
 */

/**
 * Head or face object types. Head is the new approach and should be treated as standard.
 */
type AutoDetectTypeFaceOrHead = "face" | "head";

/**
 * All the object types which can be be automatically detected.
 */
type AutoDetectType = AutoDetectTypeFaceOrHead | "licenseplate" | "person" | "vehicle" | "ID" | "screen" | "document";

/**
 * All of the types the app knows about, including (live) tracked objects.
 */
type Type = AutoDetectType | "blob" | "unknown";

/**
 * Detailed vehicle types which can be distinguished, not by the user though yet.
 */
type VehicleType = "autoRickshaw" | "bus" | "car" | "motorbike" | "motorbikeRickshaw" | "rickshaw" | "truck";

/**
 * Detailed device types which can be distinguished, not by the user though yet.
 */
type ScreenType = "Laptop" | "mobiledevice" | "tvmonitor";

/**
 * All of the types detection can ever yield or gets produced.
 */
type AnyType = Type | VehicleType | ScreenType;

/**
 * Frame number, zero based.
 */
type FrameNum = number;

/**
 * Object identifier. Used internally. Not unique per original object as reassignment is possible.
 */
type ObjectId = number;

/**
 * Unique object identifier. Not used for the application, however it allows to ensure that an object is the very one.
 */
type Uuid = string;

interface IRawObject {
    /**
     * Best frame number (the one with the highest detection score).
     */
    bestFrame?: FrameNum;
    /**
     * Best frame timestamp. Cached information to avoid frame map lookups.
     */
    bestFrameTimestamp?: number;
    /**
     * Object identifier. May have been reused due to reassignment.
     */
    id: ObjectId;
    /**
     * Unique identifier of that object.
     */
    uuid?: Uuid;
    /**
     * Links to associated objects, e.g. license plates to vehicles.
     */
    links?: Uuid[];
    /**
     * The type of the object. As it's not always auto detected it can be anything.
     */
    type: AnyType;
    /**
     * Number of first frame this object occurred at.
     */
    firstFrame: FrameNum;
    /**
     * Number of last frame this object is present. Notice: gaps might be possible!
     */
    lastFrame: FrameNum;
    /**
     * Additional attributes for this object, depending on the type, cast if needed.
     */
    attributes?: IVehicleAttributes | ILicensePlateAttributes;
}

/**
 * Object detection confidence, between 0 and 1.
 */
type ConfidenceValue = number;

/**
 * Detection compound, name of the detected item plus confidence.
 */
interface INameAndConfidence {
    name: string;
    confidence: ConfidenceValue;
}

interface IVehicleAttributes {
    type: VehicleType;
    make?: INameAndConfidence;
    model?: INameAndConfidence;
    color?: INameAndConfidence;
}

interface ILicensePlateAttributes {
    /**
     * The license plate string, if detected.
     */
    string?: INameAndConfidence;
    /**
     * Region of the license plate, if detected.
     */
    region?: INameAndConfidence;
}

type ObjectAttributes =
    ILicensePlateAttributes |
    IVehicleAttributes;

/**
 * @deprecated
 */
type ProfileNameLegacy =
    "gen6_standard" |
    "gen6_highMotion" |
    "gen6_conference" |
    "gen5_standard" |
    "gen5_standardFast" |
    "gen5_highMotion" |
    "gen5_conference";

type DefaultProfileName =
    "gen4_higherConfidence" |
    "gen4_highMotion" |
    "gen4_standardFast" |
    "gen7_standard" |
    "gen7_standardBest" |
    "gen7_highMotion";

type ConfidenceProfileName =
    "gen7_balancedConfidence" |
    "gen7_higherConfidence" |
    "gen7_lowerConfidence" |
    "gen7_veryHighConfidence" |
    "gen7_veryLowConfidence";

type ProfileName =
    DefaultProfileName |
    ConfidenceProfileName |
    /**
     * Do not use this one, for internal purposes only.
     */
    "default";

/**
 * @deprecated
 */
type ProfileNameAll =
    ProfileName |
    ProfileNameLegacy;

}
