/*******************************************************************************
 * Copyright (C) 2020 Sighthound, Inc. All rights reserved.
 * The information and source code contained herein is the
 * exclusive property of Sighthound, Inc. No part of this software
 * may be used, reproduced, stored or distributed in any form,
 * without explicit written authorization from Sighthound, Inc.
 ******************************************************************************/

declare namespace Sighthound.Api.Core.Process {

type SyncTransient = "transient";

type RequestDescriptor<F, O> = Sighthound.Api.Core.Input & Operation.IScope & {
    features: F[];
    outputUri?: string;
    outputContext?: Partial<O>;
    notifications?: Notification.IConfig[];
    tag?: Sighthound.Api.Core.AnyTag;
    acl?: string[];
    sync?: boolean | SyncTransient;
    transient?: boolean;
    stateDrop?: "skipOnError";
    void?: Partial<{
        delay: number;
    }>;
};

interface IVideoSegment {
    startTimeOffset: string;
    endTimeOffset: string;
}

interface IVideoScope {
    segments?: IVideoSegment[];
}

/**
 * Detection Configuration
 */
interface IDetectConfig {

    /**
     * Which Sighthound model to use for this detection.
     */
    model: string;

    /**
     * Include additional attributes in the output if available.
     */
    includeAttributes: boolean;

    /**
     * Include bounding box coordinates for every frame an object exists.
     */
    includeBoundingBoxes: boolean;
}

/**
 * Detection configuration.
 */
interface IDetectConfigs {

    /**
     * Face detection configuration options.
     */
    faceDetectConfig?: Partial<IDetectConfig>;

    /**
     * Person detection configuration options.
     */
    personDetectConfig?: Partial<IDetectConfig>;

    /**
     * Vehicle detection configuration options.
     */
    vehicleDetectConfig?: Partial<IDetectConfig>;

    /**
     * License plate detection configuration options.
     */
    licensePlateDetectConfig?: Partial<IDetectConfig>;
}

/**
 * The requested features to use during processing. Their presence also determines if such
 * automatic detection of objects should happen at all (skipped if nothing's declared).
 */
type DetectFeature =
    /**
     * Face Detection and Tracking
     */
    "FACE_DETECTION" |
    /**
     * Head Detection and Tracking
     */
    "HEAD_DETECTION" |
    /**
     * Person Detection and Tracking
     */
    "PERSON_DETECTION" |
    /**
     * License Plate Detection and Tracking
     */
    "LICENSE_PLATE_DETECTION" |
    /**
     * Vehicle Detection and Tracking
     */
    "VEHICLE_DETECTION" |
    /**
     * Speech transcription (if audio is present)
     */
    "SPEECH_TRANSCRIPTION" |
    /**
     * ID Detection
     */
    "ID_DETECTION" |
    /**
     * Screen Detection
     */
    "SCREEN_DETECTION" |
    /**
     * Document Detection
     */
    "DOCUMENT_DETECTION";

/**
 * REST API Response Body
 *
 * Returns an object with the full operation name, or the actual outcome already,
 * which is either a result or an error.
 */
type ResponseDescriptor<P extends Product, R extends Sighthound.Api.Core.AnyTypeOf<P>> = Partial<Operation.Outcome<P, R>> & {

    /**
     * The full name of the operation. Whether or not it is valid beyond the request
     * (e.g. for querying the status again) depends on the implementation and/or
     * configuration, i.e. it can also be pretty much a symbolic value.
     */
    name: Operation.Name;
};

}