/*******************************************************************************
 * Copyright (C) 2020 Sighthound, Inc. All rights reserved.
 * The information and source code contained herein is the
 * exclusive property of Sighthound, Inc. No part of this software
 * may be used, reproduced, stored or distributed in any form,
 * without explicit written authorization from Sighthound, Inc.
 ******************************************************************************/

declare namespace Sighthound.Api.Redactor {

/**
 * Features which are typically only needed for Redactor editor usage.
 */
type EditorFeature =
    /**
     * Prepare media. Implicit, kind of, unless it's a pure onboarding call.
     */
    "MEDIA_PREPARE" |
    /**
     * Delete state (Redactor session), which is solely for session state
     * held by the server across different Sighthound.Api.Core.Operation. It will NOT have any
     * effect on storage content. It also assumes that there is already
     * something available, but it can also be used in conjunction with
     * other features for a one-shot solution.
     */
    "STATE_DROP";

/**
 * The request media input and output features.
 */
type MediaFeature = EditorFeature |
    /**
     * @deprecated Legacy alias for MEDIA_RENDER.
     */
    "MEDIA_RENDERING" |
    /**
     * Render redacted media.
     */
    "MEDIA_RENDER";

interface IOutputContext {
    exportKey?: string;
    state: boolean;
    stateConfig?: Partial<{
        includeData: boolean,
        includeMedia: boolean,
    }>;
    speechTranscriptionData?: boolean;
    speechTranscriptionDataConfig?: Partial<{
        suffix: string;
        compress: boolean;
    }>;
}

/**
 * Language codes for speech.
 */
type LanguageCode = "cn" | "cs" | "de" | "en-us" | "es" | "fr" | "it" | "ja" | "ko" | "nl" | "pl" | "pt" | "ru";

interface ISpeechTranscriptionConfig {

    languageCode: string;

    // RESERVED (not implemented yet) ...
    diarizationSpeakerCount?: number;
    enableSpeakerDiarization?: boolean;
    enableWordConfidence?: boolean;
    filterProfanity?: boolean;
}

interface IUIProgress extends Sighthound.Api.Core.Operation.IProgress, Partial<Sighthound.Redactor.IHasSessionId> {
}

interface IPrepareConfig extends Sighthound.Redactor.ISessionDescriptor, Sighthound.Api.Core.FetchTweaks {
    hashTypes: Sighthound.Metadata.HashType[],
}

interface IContext<R extends IRenderConfig> extends Sighthound.Api.Core.Process.IDetectConfigs {

    thumbnailConfig: Partial<IThumbnailConfig>;
    renderConfig: Partial<R>;
    processConfig: Partial<IProcessConfig>;
    speechTranscriptionConfig: Partial<ISpeechTranscriptionConfig>;
    prepareConfig: Partial<IPrepareConfig>;
}
type Context = IContext<IRenderConfig>;

/**
 * Additional configuration and context for images.
 */
type IImageContext = IContext<IImageRenderConfig>;

/**
 * Additional configuration and context for videos.
 */
interface IVideoContext extends IContext<IVideoRenderConfig>, Sighthound.Api.Core.Process.IVideoScope {
}

/**
 * Thumbnail generation configuration.
 */
interface IThumbnailConfig {

    /**
     * Generate a thumbnail from the frame where an object is detected.
     */
    fullFrame: boolean;

    /**
     * Generate a thumbnail of the object.
     */
    object: boolean;
}

/**
 * @deprecated
 */
type RedactionStyleLegacySmoothBlur = "SMOOTH_BLUR";

type RedactionStyle = "OUTLINE" | "FILL" | "PIXELATE" | "MOSAIC" | "BLUR" | "KEEP" | RedactionStyleLegacySmoothBlur;
type RedactionShape = "RECTANGLE" | "ELLIPSE";
type Intensity = "LOW" | "MEDIUM" | "HIGH";
type RedactionIntensity = Intensity;
type RedactionColor = string;
type AudioMask = "SILENCE" | "BEEP" | "SCRAMBLE" | "NONE";
type AudioMaskParams = Readonly<[number, number]> | Intensity;
type VideoQuality = number;
type VideoPreset = "ULTRAFAST" | "SUPERFAST" | "VERYFAST" | "FASTER" | "FAST" | "MEDIUM" | "SLOW" | "SLOWER" | "VERYSLOW";

type VideoRange = [number, number] | { startTime: number, endTime: number};

/**
 * Redaction box presentation. Either pixel coordinates or 0..1 relative to the video resolution.
 */
interface IRedactionBox {
    x: number;
    y: number;
    width: number;
    height: number;
    relative?: boolean;
}

interface IHasRedactionTransientData {
    redactionTransientData: {
        static?: IRedactionBox[];
    }
}

type AudioRegion = Readonly<[number, number]>;

interface IRenderConfig extends IHasRedactionTransientData {
    redactionTypes: Sighthound.Analytics.Type[],
    redactionStyle: RedactionStyle;
    redactionShape: RedactionShape;
    redactionIntensity: RedactionIntensity;
    redactionColor: RedactionColor;
    exportMetadata: boolean;
    exportHashes?: Array<Sighthound.Metadata.HashType>;
}
interface IAudioRenderConfig {
    maskAudio: AudioMask;
    maskAudioParams: AudioMaskParams;
    maskAudioRegions: AudioRegion[];
}
interface IVideoRenderConfig extends IRenderConfig, IAudioRenderConfig {
    videoRange: VideoRange;
    videoQuality: VideoQuality;
    videoPreset: VideoPreset;
}
interface IImageRenderConfig extends IRenderConfig {
    imageQuality: VideoQuality;
}
type MediaRenderConfig = IVideoRenderConfig & IImageRenderConfig;

/**
 * Opaque, basic definition of what a profile looks internally.
 */
type Profile = Record<string, unknown>;

/**
 * Extension factor, either equally for every side or clockwise [top,right,bottom,left].
 */
type BoxExpansionFactor = number|[number, number, number, number];

interface IProcessConfig {

    /**
     * Detection threshold for regular (outer) objects (0..1).
     */
    objectsConfidenceThreshold: number;

    /**
     * Detection threshold for (inner) objects, i.e. located inside other objects (0..1).
     */
    innerObjectsConfidenceThreshold: number;

    /**
     * Factor for which to increase the face boxes.
     */
    faceBoxExpansionFactor: BoxExpansionFactor;

    /**
     * Factor for which to increase the head boxes.
     */
    headBoxExpansionFactor: BoxExpansionFactor;

    /**
     * Name of the detection profile to use.
     */
    profileName: Sighthound.Analytics.ProfileName;

    /**
     * Detection profile overlay, depends on the selected internal profile's details.
     */
    profileOverlay: Profile;
}

type Feature = Sighthound.Api.Core.Process.DetectFeature | MediaFeature | "VOID";

type VideoRequestDescriptor = Sighthound.Api.Core.Process.RequestDescriptor<Feature, IOutputContext> & {
    videoContext?: Partial<IVideoContext>;
};

type ImageRequestDescriptor = Sighthound.Api.Core.Process.RequestDescriptor<Feature, IOutputContext> & {
    imageContext?: Partial<IImageContext>;
};

interface IVideoDescriptor extends Sighthound.Redactor.ISessionDescriptor {
    rendered: boolean;
}

type ProductType = "redactor";

type Exported = {
    size: number;
} & Partial<{
    [key in `${Sighthound.Metadata.HashType}`]: string;
}>;

type StoredProcessResult = Sighthound.Api.Core.Operation.IStoredProcessResult & {
    exported?: Exported,
};

type DeleteDescriptor = Sighthound.Api.Core.Input;

type RequestDescriptor = Sighthound.Api.Core.Process.RequestDescriptor<Feature, IOutputContext>;

type ResponseDescriptor = Sighthound.Api.Core.Process.ResponseDescriptor<ProductType, Sighthound.Api.Core.Operation.StoredProcessResponse<Sighthound.Api.Version11, ProductType, StoredProcessResult>>;

type OperationStatus = Sighthound.Api.Core.Operation.Status<ProductType, Sighthound.Api.Core.Operation.StoredProcessResponse<Sighthound.Api.Version11, ProductType, StoredProcessResult>>;

}
