/*******************************************************************************
 * Copyright (C) 2024 Sighthound, Inc. All rights reserved.
 * The information and source code contained herein is the
 * exclusive property of Sighthound, Inc. No part of this software
 * may be used, reproduced, stored or distributed in any form,
 * without explicit written authorization from Sighthound, Inc.
 ******************************************************************************/

declare namespace Sighthound.Redactor {

type SessionId = string;

interface IHasSessionId {
    sessionId: Sighthound.Redactor.SessionId;
}

interface IHasMetadata {
    metadata: MetadataCollection;
}

interface ISessionDescriptor extends Sighthound.Redactor.IHasSessionId, IHasMetadata {
}

/**
 * To which session specifically an embedded Redactor instance should attach to.
 */
interface IWhereToProject extends Sighthound.Redactor.IHasSessionId {
    /**
     * The associated project identifier. Even though sessions identifiers are
     * supposed to be globally unique it might be required to provide this for
     * proper resolution.
     */
    projectId: Sighthound.Api.ProjectId;

    /**
     * The user name to use in the opening of the session, to preserve ownership.
     */
    userName?: string;
}

type WhereTo = Sighthound.Redactor.SessionId | IWhereToProject;

/**
 * Commands which can be sent to the embedded editor in the browser, to open the
 * export dialog, close the currently open video/session or destroy the whole
 * instance (meaning proper shutdown of pending messages, DOM event handlers, etc).
 */
type Command = "export" | "close" | "destroy";

/**
 * What an embedded Redactor instance represents. This is the public interface and
 * should be the one and only way to interact with it. Any undocumented features
 * might change in the future and cannot be counted on to be present or having the same
 * behavior!
 */
interface IEditor {

    /**
     * To check whether or not a video is in the works.
     *
     * @returns True if video editing happens, false if there is none loaded and
     * we're back in the list view. Undefined if the instance is invalid.
     */
    active: () => boolean|undefined,

    /**
     * To quit/close the currently opened video.
     *
     * @returns True if the video has been closed, false if none was open.
     */
    cleanup: () => Promise<boolean>,

    /**
     * Start to destroy the current instance. Recommended to be called before the whole
     * surrounding environment is going away. Does things like detaching event handlers
     * and ending server communication channels.
     */
    destroy: () => void;

    /**
     * Start rendering the editor, i.e. mounting it in the DOM, starting server comm etc.
     * This allows a deferred creation point, apart from an instance's construction time.
     *
     * @param cb - Invoked if the editor is ready, i.e. if the initial render call has been
     * committed. If an error occurred the instance must be treated as invalid
     * and should be destroyed.
     */
    renderEditor: (cb?: (err?: unknown) => void) => void,

    /**
     * @see renderEditor
     * @throws The err(or) otherwise being reported in the readiness callback.
     */
    renderEditorAsync: () => Promise<void>,

    /**
     * Send a command to the active editor instance.
     *
     * @param command - The command to execute.
     * @param args - Arguments for the command. It may contain a session identifier in case
     * the video is not opened.
     *
     * @returns True if the command was accepted, false if e.g. the editor is not open and
     * a close command had been issued.
     */
    sendCommand: (command: Command, args?: Partial<Sighthound.Redactor.IHasSessionId>) => boolean;

    /**
     * In which language the editor instance should appear. Can switch at runtime.
     *
     * @param locale - New locale, default is `en-US`.
     */
    setLocale: (locale: string) => void,
}

type CustomizeShowHide = "show" | "hide";
type CustomizeShowHideClose = CustomizeShowHide | "close";
type CustomizeShowHideEventOnly = CustomizeShowHide | "eventOnly";

interface ICustomize {
    export?: Partial<{
        download: CustomizeShowHideEventOnly,
        save: CustomizeShowHideClose,
    }>,
    session?: Partial<{
        auditLogs: CustomizeShowHide,
        folders: CustomizeShowHide,
        remove: CustomizeShowHideEventOnly,
        deleteRedactionData: CustomizeShowHide,
    }>,
    videolist?: Partial<{
        customMetadataKeys: string[],
    }>
}

interface ICustomizable {
    customize?: ICustomize;
}

type MetadataCollection = Sighthound.Metadata.Collection<ICustomize>;

interface IEditorOptions extends ICustomizable {

    cssHref?: Partial<{

        /** CSS overwrites for the main instance, to turn into `<style>@import url( '{ITEM}' )</style>`. */
        main: string|string[];

        /** CSS injection (into `<head>`) for anything font, turns into `<link href={ITEM} rel=stylesheet.../>`. */
        fonts: string|string[]|null;
    }>;

    /** True to host the editor in its own iFrame, remoting calls to and receiving events from it. */
    iframed?: boolean;

    /** The initial locale/language to use. Default is `en-US`. */
    locale: string;

    /** The desired log level for browser console output. */
    logLevel: Sighthound.Logging.Level;

    /** What prefix to use for log messages. Helps with easier filtering in the console. */
    logPrefix?: string;

    /**
     * True to send logs to the server to be stored amongst backend messages. Can help to
     * identify issues with certain user interactions or client behavior in general.
     */
    logToServer?: boolean;

    /**
     * True to place the whole editor instance in a shadow DOM. This helps to isolate it from the
     * outer parts (styles, event bubbling, etc). Recommended, ineffective though when `iframed: true`.
     */
    shadowed?: boolean;

    /**
     * URL pointing to an overlay translation file in HJSON format, e.g. for rewording or customer
     * provided proprietary translations (as long as the language type is known/supported).
     */
    translationOverwritesUrl?: string;
}

/**
 * Embedded Redactor instances are getting created by this class, *once* for the
 * whole browser/DOM space. It *strictly* implements the public interface, no
 * private or extra properties.
 *
 * @see IEditor
 */
class SighthoundRedactorEditor implements IEditor {

    /**
     * @param htmlElement - The HTML element (DOM node) hosting the editor instance.
     * @param serverUrl - The server URL where Redactor is hosted at, helps with proxies etc.
     * @param whereTo - Which session (if any) to open initially.
     * @param pOptions - Options for the instance.
     */
    public constructor(
        htmlElement: HTMLElement,
        serverUrl: string,
        whereTo: WhereTo,
        pOptions?: Partial<IEditorOptions>);

    active: () => boolean|undefined;
    cleanup: () => Promise<boolean>;
    destroy: () => void;
    renderEditor: (cb?: (err?: unknown) => void) => void;
    renderEditorAsync: () => Promise<void>;
    sendCommand: (command: Command, args?: Partial<Sighthound.Redactor.IHasSessionId>) => boolean;
    setLocale: (locale: string) => void;
}

type IEventDetail = Sighthound.Redactor.IHasSessionId;

interface IEventDetailMany {
    /** Multiple videos/sessions have been selected. */
    sessionIds: Sighthound.Redactor.SessionId[];
}

interface IExportDownloadEventDetail extends IEventDetail {
    /** Where a download is offered from. */
    readonly url: string;
}

interface IExportSubmitEventDetail<O> extends IEventDetail {
    /**
     * Internal redaction options, ultimately.
     * @deprecated  Not useful for the recipient since they require translation or
     *              mapping into a rendering configuration.
     */
    readonly redactionOptions: O;

    /** The ready-to-use rendering configuration for API usage. */
    readonly renderConfig: Sighthound.Redactor.IRenderConfig;
}

interface IEvent<T extends string, D> extends Sighthound.Redactor.IHasSessionId {
    /** Event type (string). */
    readonly type: T;
    /** Event details. */
    readonly detail: D;
    /** Associated/active user name causing the event to happen. */
    readonly userName: string;
}

/** Event: download button/menu has been clicked on. */
type ExportDownloadEvent = IEvent<"export.download", IExportDownloadEventDetail>;

/** Event: export button/menu has been clicked on. Options are provided. */
type ExportSubmitEvent = IEvent<"export.submit", IExportSubmitEventDetail<unknown>>;

/** Event: any cancel button/method was clicked/invoked. */
type ExportCancelEvent = IEvent<"export.cancel", IEventDetail>;

/** Event: the save button was clicked on. */
type ExportSaveEvent = IEvent<"export.save", IEventDetail>;

/** Event: video removal menu item was clicked on. */
type SessionRemoveEvent = IEvent<"session.remove", IEventDetail>;

/** Event: multiple videos were selected and removal requested. */
type SessionRemoveManyEvent = IEvent<"session.remove.many", IEventDetailMany>;

/** Event: redaction data drop menu was clicked on (in the video list). */
type SessionDeleteRedactionDataEvent = IEvent<"session.delete_redaction_data", IEventDetail>;

/** Event: multiple videos were selected and redaction data drop was requested. */
type SessionDeleteRedactionDataManyEvent = IEvent<"session.delete_redaction_data.many", IEventDetailMany>;

/** Event: the editor just got opened (a video was loaded into it). */
type EditorEnterEvent = IEvent<"editor.enter", IEventDetail>;

/** Event: the editor just got closed and the UI is back to the video list view, if any. */
type EditorExitEvent = IEvent<"editor.exit", IEventDetail>;

}
